/**
    \file: bgt24mtr12.c
    \author: Assad Ali
  	\brief: File implements BGT base functions to configure the user settings via SPI Data Bits in the control register of the BGT24MTR12.

 */

/* ===========================================================================
 ** Copyright (C) 2017-2018 Infineon Technologies AG
 ** All rights reserved.
 ** ===========================================================================
 **
 ** ===========================================================================
 ** This document contains proprietary information of Infineon Technologies AG.
 ** Passing on and copying of this document, and communication of its contents
 ** is not permitted without Infineon's prior written authorization.
 ** ===========================================================================
 */

/*
==============================================================================
   1. INCLUDE FILES
==============================================================================
 */
#include "Config_PORT.h"
#include "iodefine.h"

#include "bgt24mtr12.h"
#include "LMX249x.h"
#include "peripheral.h"
#include "config.h"
#include "radar.h"

/*
==============================================================================
   2. DATA
==============================================================================
 */

static volatile  uint16_t   g_bgt_conf = BGT24_BASE_CONF;

volatile  uint16_t   bgt_ana_command = 2;

volatile  uint16_t   bgt_ana_out[3] = {0,0,0};

volatile  float  g_bgt_vout_tx_power = 0;

volatile  float  g_bgt_temperature_C = 0;

/*
==============================================================================
   3. API FUNCTIONS
==============================================================================
 */

void bgt_init(void)
{
	g_bgt_conf = BGT24_BASE_CONF;

	if (LNA_GAIN_ENABLE)
	{
		g_bgt_conf &= BGT24_ENA_MASK;
	}
	else
	{
		g_bgt_conf |= BGT24_DIS_MASK;
	}

	bgt_set_tx_power((uint8_t)BGT_TX_POWER);		// configure BGT24

	bgt_ana_vref_tx();
}

//------------------------------------------//

/*
 * start the transmitter on the BGT24MTR11
 */
void bgt_start_tx(void)
{
	g_bgt_conf &= BGT24_ENA_PA_MASK;

	bgt_set_config(g_bgt_conf);

}  // end of BGTStartTX()

//------------------------------------------//

/*
 * stop the transmitter on the BGT24MTR11
 */
void bgt_stop_tx(void)
{
	g_bgt_conf |= BGT24_DIS_PA_MASK;

	bgt_set_config(g_bgt_conf);

}  // end of BGTStopTX()

//------------------------------------------//

void bgt_ana_temp(void)
{
	bgt_ana_command = 0;

	g_bgt_conf &= BGT24_AMUX_VOUT_TX;

	g_bgt_conf |= BGT24_AMUX_2;
}

//------------------------------------------//

void bgt_ana_vout_tx(void)
{
	bgt_ana_command = 1;

	g_bgt_conf &= BGT24_AMUX_VOUT_TX;
}

//------------------------------------------//

void bgt_ana_vref_tx(void)
{
	bgt_ana_command = 2;

	g_bgt_conf &= BGT24_AMUX_VOUT_TX;

	g_bgt_conf |= BGT24_AMUX_0;
}

//------------------------------------------//

/*
 * Enable the LNA on the BGT24MTR12
 */
void bgt_lna_gain_enable(void)
{
	g_bgt_conf &= BGT24_ENA_MASK;
}

//------------------------------------------//

/*
 * Disable the LNA on the BGT24MTR12
 */
void bgt_lna_gain_disable(void)
{
	g_bgt_conf |= BGT24_DIS_MASK;
}

//------------------------------------------//

/*
 * Status (Enable or Disable) of the LNA on the BGT24MTR12
 */
uint8_t bgt_lna_gain_is_enable(void)
{
	return (uint8_t)((g_bgt_conf & BGT24_DIS_MASK) == 0 ? true : false);
}

//------------------------------------------//

/*
 * get 16-bit SPI values to the BGT24MTR12
 */
uint16_t bgt_get_config(void)
{
	return g_bgt_conf;
}

//------------------------------------------//

/*
 * get the BGT Tx power level from [0 - 7]
 */
uint8_t bgt_get_tx_power(void)
{
	return (uint8_t)(g_bgt_conf & 0x07);	// lower byte contains the Tx power levels info
}

//------------------------------------------//

/*
 * set the BGT Tx power level from [0 - 7]
 */
void bgt_set_tx_power(uint8_t power_level)
{
	g_bgt_conf &= 0xFFF8;		// clears the last 3-bits

	switch (power_level)
	{
	case 1:
		g_bgt_conf |= BGT24_PC_PA_7;	// Reduction by 9dBm
		break;

	case 2:
		g_bgt_conf |= BGT24_PC_PA_6;  // Reduction by 6dBm
		break;

	case 3:
		g_bgt_conf |= BGT24_PC_PA_5;  // Reduction by 4dBm
		break;

	case 4:
		g_bgt_conf |= BGT24_PC_PA_4;  // Reduction by 2.5dBm
		break;

	case 5:
		g_bgt_conf |= BGT24_PC_PA_3;  // Reduction by 1.4dBm
		break;

	case 6:
		g_bgt_conf |= BGT24_PC_PA_2;  // Reduction by 0.8dBm
		break;

	case 7:
	default:
		g_bgt_conf |= BGT24_PC_PA_0;  // TX on with maximum power

	}

	bgt_ana_vout_tx();
}

//------------------------------------------//

void bgt_power_up(void)
{
//	DIGITAL_IO_SetOutputLow(&DIGITAL_IO_BGT_POWER_ENABLE);
}

//------------------------------------------//

void bgt_power_down(void)
{
//	DIGITAL_IO_SetOutputHigh(&DIGITAL_IO_BGT_POWER_ENABLE);
}

//------------------------------------------//
void bgt_lowest_power_with_q2_disable(void)
{
	bgt_set_config((uint16_t)BGT24_POWER_CONF);
}

/*
 * send 16-bit SPI values to the BGT
 */
void bgt_set_config(uint16_t SPIdata)
{
	uint16_t shuffled_data = 0;

	shuffled_data  = (SPIdata << 8);		// lower byte goes to upper byte position

	shuffled_data |= (SPIdata >> 8);		// upper byte goes to lower byte position

	spi_transmit_data_bgt( (uint8_t *)&shuffled_data , 2U/*, (void*) &DIGITAL_IO_SPI_M_CS_BGT24*/);
}

