/**
    \file: pll_lmx249x.c
    \author: Assad Ali, Thomas Finke
        \brief: This file implements the function declarations for Texas Instrument's PLL named LMX249x
 */

/* ===========================================================================
 ** Copyright (C) 2017 Infineon Technologies AG
 ** All rights reserved.
 ** ===========================================================================
 **
 ** ===========================================================================
 ** This document contains proprietary information of Infineon Technologies AG.
 ** Passing on and copying of this document, and communication of its contents
 ** is not permitted without Infineon's prior written authorization.
 ** ===========================================================================
 */

/*
==============================================================================
   1. INCLUDE FILES
==============================================================================
*/
#include "Config_PORT.h"
#include "iodefine.h"
//#include "firmware_api.h"
#include "pll.h"
//#include "mcu.h"
//#include "defines.h"

#include "config.h"
#include "common.h"
#include "peripheral.h"
#include "radar.h"


/*
==============================================================================
   2. DATA
==============================================================================
*/

LMX249x_Object_s gLmx249x_pll;

void* pll_handle = &gLmx249x_pll;

LMX249x_FracOrder_e  g_pll_frac_order = LMX249x_FRAC_ORD_SECOND;

volatile  uint32_t   g_pll_base_freq = FMCW_BASE_FREQ_MHz;

volatile  float  g_bandwidth_MHz = BANDWIDTH_MHz;

volatile  uint32_t   g_chirp_time_us = CHIRP_TIME_us;

volatile uint32_t   g_inter_chirp_time_us = INTER_CHIRP_TIME_us;

double fskFreq=0.5;

/*
==============================================================================
   3. API FUNCTIONS
==============================================================================
*/

void pll_init(void* pll_handle)
{
        LMX249x_Object_s* lmx249x_pll = (LMX249x_Object_s*)pll_handle;
        LMX249x_HardwareSetup_s s_sPLLHardwareSetup;

        /* send general configuration to PLL */
        /* --------------------------------- */
        s_sPLLHardwareSetup.dReferenceFreq         = 40.0;
        s_sPLLHardwareSetup.uReferenceDivider      = 1;
        s_sPLLHardwareSetup.eReferenceDoubler      = LMX249x_OSCIN_DOUBLER_OFF;
        s_sPLLHardwareSetup.eOscInMode             = LMX249x_OSCIN_SINGLE_ENDED;
        s_sPLLHardwareSetup.uExternalDivider       = 16;
        s_sPLLHardwareSetup.eTrig1PinFunction      = LMX249x_MUX_IN_TRIG1;
        s_sPLLHardwareSetup.eTrig1PinDriveMode     = LMX249x_PIN_FCN_INPUT;
        s_sPLLHardwareSetup.eTrig2PinFunction      = LMX249x_MUX_IN_TRIG2;
        s_sPLLHardwareSetup.eTrig2PinDriveMode     = LMX249x_PIN_FCN_INPUT;
        s_sPLLHardwareSetup.eModPinFunction        = LMX249x_MUX_OUT_FLAG1_FROM_RAMP;
        s_sPLLHardwareSetup.eModPinDriveMode       = LMX249x_PIN_FCN_PULLUPDN_OUT;
        s_sPLLHardwareSetup.eMUXoutPinFunction     = LMX249x_MUX_OUT_READ_BACK;
        s_sPLLHardwareSetup.eMUXoutPinDriveMode    = LMX249x_PIN_FCN_PULLUPDN_OUT;
        s_sPLLHardwareSetup.eChargePumpPolarity    = LMX249x_CPPOL_NEGATIVE;
        s_sPLLHardwareSetup.uChargePumpCurrent     = 25;
        s_sPLLHardwareSetup.uChargePumpCurrentFS   = 0;
        s_sPLLHardwareSetup.uChargePumpThresholdLo = 4;
        s_sPLLHardwareSetup.uChargePumpThresholdHi = 46;
        s_sPLLHardwareSetup.eChargePumpPulseWidth  = LMX249x_CP_PULSE_860PS;
        s_sPLLHardwareSetup.eCycleSlipReduction    = LMX249x_CSR_DISABLED;
        s_sPLLHardwareSetup.uFastLockTimer         = 0;
        s_sPLLHardwareSetup.uLockDetectNumGoodEdge = 32;
        s_sPLLHardwareSetup.uLockDetectNumBadEdge  = 4;
        s_sPLLHardwareSetup.eLockDetectWindow      = LMX249x_DLD_TOL_1NS;

        LMX249x_init(lmx249x_pll, &s_sPLLHardwareSetup, spi_transmit_data_pll/*, (void*)&DIGITAL_IO_SPI_M_CS_PLL*/);

        LMX249x_setPowerState(lmx249x_pll, LMX249x_POWER_CE);
}

void pll_configure_ramps(void* pll_handle)
{
        // setup data configuration data for PLL chip
        LMX249x_Object_s* lmx249x_pll = (LMX249x_Object_s*)pll_handle;

        LMX249x_RampGlobal_s pllGlobalRampSetup;

        pllGlobalRampSetup.dBaseFrequency = g_pll_base_freq;                                                    /*!< base frequency */
//        pllGlobalRampSetup.dDeviationFrequency = 30.0 + g_pll_base_freq;                                /*!< FSK deviation frequency */
        pllGlobalRampSetup.dDeviationFrequency = fskFreq + g_pll_base_freq;                                /*!< FSK deviation frequency */
        pllGlobalRampSetup.dMinFrequency = ((double)D2GO_MIN_FREQUENCY_kHz * 0.001f);                                                                   /*!< Minimum frequency that can't be underrun by a ramp */
        pllGlobalRampSetup.dMaxFrequency = ((double)D2GO_MAX_FREQUENCY_kHz * 0.001f);                                                                   /*!< Maximum frequency that can't be overrun by a ramp */
        pllGlobalRampSetup.dComp0Freq = ((double)D2GO_MIN_FREQUENCY_kHz * 0.001f);                                                                      /*!< Frequency that is used by Comparator 0 to compare the ramp with. */
        pllGlobalRampSetup.dComp1Freq = ((double)D2GO_MAX_FREQUENCY_kHz * 0.001f);                                                                      /*!< Frequency that is used by Comparator 0 to compare the ramp with. */
        pllGlobalRampSetup.eTriggerA = LMX249x_RAMP_TRIG_TRIG2_TERMINAL_RISING_EDGE;    /*!< Define the source for Trigger A */
        pllGlobalRampSetup.eTriggerB = LMX249x_RAMP_TRIG_TRIG1_TERMINAL_RISING_EDGE;    /*!< Define the source for Trigger B */
        pllGlobalRampSetup.eTriggerC = LMX249x_RAMP_TRIG_NEVER_TRIGGERS;                        /*!< Define the source for Trigger C */
        pllGlobalRampSetup.eRampClock = LMX249x_RAMP_CLOCK_INTERNAL;                            /*!< determines if internal or external clock for ramp timing is used */
        pllGlobalRampSetup.eModulation = LMX249x_RAMP_MODULATION_FM;                            /*!< Kind of modulation */
        pllGlobalRampSetup.eAutoOff = LMX249x_RAMP_AUTO_OFF_DISABLE;                            /*!< Defines, if the Ramp should be turned off after a certain amount of ramps */
        pllGlobalRampSetup.uNumRamps = g_num_of_chirps * 3;                                     /*!< Number of ramps to do before ramp is disabled (set to 0 for infinite number of ramps) */
        pllGlobalRampSetup.eRampCountTrigger = LMX249x_RAMP_NEXT_TRIG_TRIGB;                    /*!< Defines the increment trigger for the ramp counter */
        pllGlobalRampSetup.eDevTrigger = LMX249x_RAMP_NEXT_TRIG_TRIGA;                          /*!< Defines the deviation trigger for FSK operation */
        pllGlobalRampSetup.eFracOrder = g_pll_frac_order;
        pllGlobalRampSetup.eDitherMode = LMX249x_FRAC_DITHER_STRONG;

        LMX249x_RampSection_s pllRampSections[4];

        /* reset ramp  */
        pllRampSections[0].dFreqShift   = 0.0;
        pllRampSections[0].eFastlock    = LMX249x_RAMP_FASTLOCK_DISABLED;
        pllRampSections[0].dTramp       = 0;
        pllRampSections[0].eFlag        = LMX249x_RAMP_FLAG_CLR_FLAG;
        pllRampSections[0].eReset       = LMX249x_RAMP_RST_ENABLE;                                      /* first ramp resets counters */
        pllRampSections[0].eNextTrig    = LMX249x_RAMP_NEXT_TRIG_TRIGB;
        pllRampSections[0].uNext        = 1;
        pllRampSections[0].eComparators = LMX249x_RAMP_NO_COMPARATOR;

        /* ramp up */
        pllRampSections[1].dFreqShift   = g_bandwidth_MHz;
        pllRampSections[1].eFastlock    = LMX249x_RAMP_FASTLOCK_DISABLED;
        pllRampSections[1].dTramp       = g_chirp_time_us;
        pllRampSections[1].eFlag        = LMX249x_RAMP_FLAG_SET_BOTH;
        pllRampSections[1].eReset       = LMX249x_RAMP_RST_DISABLE;
        pllRampSections[1].eNextTrig    = LMX249x_RAMP_NEXT_TRIG_RAMPX_LEN;
        pllRampSections[1].uNext        = 2;
        pllRampSections[1].eComparators = LMX249x_RAMP_NO_COMPARATOR;

        /* ramp down */
        pllRampSections[2].dFreqShift   = -g_bandwidth_MHz;
        pllRampSections[2].eFastlock    = LMX249x_RAMP_FASTLOCK_DISABLED;
        pllRampSections[2].dTramp       = 250;
        pllRampSections[2].eFlag        = LMX249x_RAMP_FLAG_SET_BOTH;
        pllRampSections[2].eReset       = LMX249x_RAMP_RST_DISABLE;
        pllRampSections[2].eNextTrig    = LMX249x_RAMP_NEXT_TRIG_RAMPX_LEN;
        pllRampSections[2].uNext        = 3;
        pllRampSections[2].eComparators = LMX249x_RAMP_NO_COMPARATOR;

        /* inter chirp */
        pllRampSections[3].dFreqShift   = 0;
        pllRampSections[3].eFastlock    = LMX249x_RAMP_FASTLOCK_DISABLED;
        pllRampSections[3].dTramp       = g_inter_chirp_time_us;
        pllRampSections[3].eFlag        = LMX249x_RAMP_FLAG_SET_BOTH;
        pllRampSections[3].eReset       = LMX249x_RAMP_RST_DISABLE;
        pllRampSections[3].eNextTrig    = LMX249x_RAMP_NEXT_TRIG_RAMPX_LEN;
        pllRampSections[3].uNext        = 0;
        pllRampSections[3].eComparators = LMX249x_RAMP_NO_COMPARATOR;


        LMX249x_configureRamps(lmx249x_pll, &pllGlobalRampSetup, pllRampSections, 4);
}

void pll_enable_ramps(void* pll_handle)
{
        LMX249x_Object_s* lmx249x_pll = (LMX249x_Object_s*)pll_handle;
        LMX249x_enableRamps(lmx249x_pll, true);
}

void pll_disable_ramps(void* pll_handle)
{
        LMX249x_Object_s* lmx249x_pll = (LMX249x_Object_s*)pll_handle;
        LMX249x_enableRamps(lmx249x_pll, false);
}

void pll_trigger_ramp(void)
{
        // wait until PLL is ready to generate the next chirp
        while (PLL_MOD != 0);

        // now start the chirp
//        PLL_TRIG1=1;
}

void pll_release_ramp_trigger(void)
{
 //       PLL_TRIG1=0;
}

void pll_trigger_deviation(void)
{
        //DIGITAL_IO_SetOutputHigh(&DIGITAL_IO_PLL_TRIG2);
}

void pll_release_deviation_trigger(void)
{
        //DIGITAL_IO_SetOutputLow(&DIGITAL_IO_PLL_TRIG2);
}

void pll_enable(void)
{
        PLL_CE=1;
}

void pll_disable(void)
{
        PLL_CE=0;
}

void pll_update_configuration(void)
{
        pll_disable_ramps(pll_handle);

        pll_configure_ramps(pll_handle);

        pll_enable_ramps(pll_handle);
}

void setFskFreq(double x1f, void* pll_handle)
{
	fskFreq = x1f;
	pll_update_configuration();
}

